<?php
/**
 * BwTransifex Component
 *
 * BwTransifex installer class for the component
 *
 * @version 1.0.1
 * @package BwTransifex
 * @subpackage BwTransifex Component Admin
 * @author Romana Boldt
 * @copyright (C) 2025 Boldt Webservice <forum@boldt-webservice.de>
 * @support https://www.boldt-webservice.de/en/forum-en/forum/bwtransifex.html
 * @license GNU/GPL, see LICENSE.txt
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

// No direct access
defined('_JEXEC') or die('Restricted access');

use Joomla\CMS\Factory;
use Joomla\Filesystem\Path;
use Joomla\Filesystem\Folder;
use Joomla\Filesystem\File;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Installer\InstallerAdapter;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Uri\Uri;
use Joomla\Database\DatabaseDriver;
use Joomla\Registry\Registry;

/**
 * Define the BwTransifex installation class
 *
 * @package BwTransifex Admin
 *
 * @since 1.0.0
 */
class com_bwtransifexInstallerScript
{
    /**
     * @var InstallerAdapter $parentInstaller
     *
     * @since 1.0.0
     */
    public InstallerAdapter $parentInstaller;

    /**
     * @var string $minimum_php_version
     *
     * @since 1.0.0
     */
    private string $minimum_php_version = "8.1";

    /**
     * @var string $minimum_joomla_release
     *
     * @since 1.0.0
     */
    private string $minimum_joomla_release = "4.3";

    /**
     * @var string release
     *
     * @since 1.0.0
     */
    private string $release = '1.0.0';

    /**
     * @var DatabaseDriver
     *
     * @since 1.0.0
     */
    private DatabaseDriver $db;

    /**
     * BwTransifex component installation preflight function
     * Called before any type of action
     *
     * @param string             $type   Which action is happening (install|uninstall|discover_install|update)
     * @param   InstallerAdapter $parent The object responsible for running this script
     *
     * @return  boolean  True on success
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    public function preflight(string $type, InstallerAdapter $parent): bool
    {
        $app     = Factory::getApplication();
        $session = $app->getSession();

//        $this->parentInstaller = $parent->getParent();
        $manifest              = $parent->getManifest();

        // Get component manifest file version
        $this->release = (string) $manifest->version;
        $session->set('release', $this->release, 'bwtransifex');

        if ($type !== 'uninstall')
        {
            // Abort if the minimum Joomla version is not met
            if (version_compare(JVERSION, $this->minimum_joomla_release, 'lt'))
            {
                $app->enqueueMessage(Text::sprintf('COM_BWTRANSIFEX_INSTALLATION_ERROR_JVERSION',
                    $this->minimum_joomla_release), 'error');

                return false;
            }

            // Abort if the minimum php version is not met
            if (version_compare(phpversion(), $this->minimum_php_version, 'lt'))
            {
                $app->enqueueMessage(Text::sprintf('COM_BWTRANSIFEX_INSTALLATION_ERROR_PHP_VERSION', $this->minimum_php_version), 'error');

                return false;
            }

            // Abort if the component being installed is older than the currently installed version
            // In some rare cares Joomla provides type update, nevertheless the extension is not installed. See
            // workaround below:
            if ($type === 'update')
            {
                $oldRelease = $this->getManifestVar('version');

                // Workaround if type is update without installed extension. In this case the method getManifestVar()
                // returns -1, which could be interpreted as 'Installed version is not newer than version to install'
                if ($oldRelease === '-1') {
                    return true;
                }

                $app->setUserState('com_bwtransifex.update.oldRelease', $oldRelease);

                if (version_compare($this->release, $oldRelease, 'lt'))
                {
                    $app->enqueueMessage(Text::sprintf('COM_BWTRANSIFEX_INSTALLATION_ERROR_INCORRECT_VERSION_SEQUENCE',
                        $oldRelease, $this->release), 'error');

                    return false;
                }
            }
        }

        return true;
    }

    /**
     * BwTransifex component installation function
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    public function install(): void
    {
        $this->showFinished(false);
    }

    /**
     * BwTransifex component uninstall function
     *
     * @since 1.0.0
     */
    public function uninstall(): void
    {
    }

    /**
     * BwTransifex component update function
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    public function update(): void
    {
        $this->showFinished(true);
    }

    /**
     * BwTransifex component installation postflight function
     *
     * @param   string            $type    type of installation

     * @return bool
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    public function postflight(string $type): bool
    {
        if ($type == 'install')
        {
            // Set BwTransifex default settings in the extensions table at install
            $this->setComponentParams(array());
        }

        if ($type == 'install' || $type == 'update')
        {
            // Check if takeover is possible
            $app  = Factory::getApplication();
            $lang = Factory::getApplication()->getLanguage();

            // First load english files
            $lang->load('com_bwtransifex.sys', JPATH_ADMINISTRATOR, 'en_GB', true);
            $lang->load('com_bwtransifex', JPATH_ADMINISTRATOR, 'en_GB', true);

            // The load current language
            $lang->load('com_bwtransifex.sys', JPATH_ADMINISTRATOR, null, true);
            $lang->load('com_bwtransifex', JPATH_ADMINISTRATOR, null, true);

            $takeoverCheck = $this->checkTakeover();

            // Do the takeover, if possible
            if ($takeoverCheck['error'] !== '')
            {
            $message = Text::_('COM_BWTRANSIFEX_TAKEOVER_GENERAL_ERROR') . "\n";
                $message .= $takeoverCheck['error'];
                $app->enqueueMessage($message, 'warning');
            }
            else
            {
                // Takeover is possible
                if ($takeoverCheck['noBwTxTables'] === true
                    && $takeoverCheck['cTxTablesExists'] === true)
                {
                    $this->doTakeover($takeoverCheck);
                }
                // Takeover is not possible
                elseif ($takeoverCheck['noBwTxTables'] === false
                    && ($takeoverCheck['cTxTablesExists'] === true
                        || $takeoverCheck['cTxComponentExists'] === true
                        || $takeoverCheck['cTxArtifactsExists'] === true
                        )
                    )
                {
                    $message = Text::_('COM_BWTRANSIFEX_TAKEOVER_PROJECTS_NOT_EMPTY_ERROR') . "\n";
                    $app->enqueueMessage($message, 'warning');
                }
            }
        }

        return true;
    }

    /**
     * Method to get the database object depending on Joomla! main version
     *
     * @return mixed
     *
     * @since 1.0.0
     */
    protected function getDb(): mixed
    {
        if (version_compare(JVERSION, '4.0.0', 'ge'))
        {
            $db = Factory::getContainer()->get('db');
        }
        else
        {
            $db = Factory::getDbo();
        }

        return $db;
    }

    /**
     * get a variable from the manifest file (actually, from the manifest cache).
     *
     * @param string $name
     *
     * @return  string
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    private function getManifestVar(string $name): string
    {
        $manifest = array();
        $db       = $this->getDb();
        $query    = $db->getQuery(true);

        $query->select($db->quoteName('manifest_cache'));
        $query->from($db->quoteName('#__extensions'));
        $query->where($db->quoteName('element') . " = " . $db->quote('com_bwtransifex'));

        try
        {
            $db->setQuery($query);

            $result   = $db->loadResult();

            // Workaround if type is update without installed extension (no entry in table). In this case we should
            // return a special value (here -1), which has to be processed in the calling method.
            if (is_null($result))
            {
                return '-1';
            }

            $manifest = json_decode($result, true);
        }
        catch (RuntimeException $e)
        {
            Factory::getApplication()->enqueueMessage($e->getMessage(), 'error');
        }

        return $manifest[$name];
    }

    /**
     * Preset default values for params of component at extensions table
     *
     * @param   array|string  $defaultParams  , may be an array or a JSON string
     *
     * @return  void
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    private function setComponentParams(array|string $defaultParams): void
    {
        if (!is_string($defaultParams))
        {
            if (count($defaultParams) === 0)
            {
                // Set default params
            }
            $params = json_encode($defaultParams);

        }
        else
        {
            $params = $defaultParams;
        }

        $db    = $this->getDb();
        $query = $db->getQuery(true);

        $query->update($db->quoteName('#__extensions'));
        $query->set($db->quoteName('params') . " = " . $db->quote($params));
        $query->where($db->quoteName('element') . " = " . $db->quote('com_bwtransifex'));

        try
        {
            $db->setQuery($query);
            $db->execute();
        }
        catch (RuntimeException $e)
        {
            Factory::getApplication()->enqueueMessage($e->getMessage(), 'error');
        }
    }

    /**
     * Check if the takeover is possible and what to take over is possible
     *
     * That is the case, if the project table of BwTransifex is empty and
     * - component ctransifex exists
     * - component ctransifex does not exist, but table #__ctransifex_projects exists. In this case take over of options
     * of the component is not possible
     *
     * @return array
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    private function checkTakeover(): array
    {
        $result = array(
            'noBwTxTables'       => false,
            'cTxComponentExists' => false,
            'cTxTablesExists'    => false,
            'cTxArtifactsExists' => false,
            'error'              => '',
        );
        $db  = $this->getDb();
        $this->db = $db;
        $query = $db->getQuery(true);

        // component ctransifex exists?
        $query->select('COUNT(' . $db->quoteName('extension_id') . ')');
        $query->from($db->quoteName('#__extensions'));
        $query->where($db->quoteName('element') . " LIKE '%com_ctransifex%'");

        $db->setQuery($query);

        $com_ctransifex = $db->loadResult();

        if ($com_ctransifex)
        {
            $result['cTxComponentExists'] = true;
        }

        try
        {
            // Check if BwTransifex has no entry for a project
            $query->clear();

            $query->select('COUNT(' . $db->quoteName('id') . ')');
            $query->from($db->quoteName('#__bwtransifex_projects'));

            $db->setQuery($query);

            $bwTransifexResult = $db->loadResult();

            if ((int)$bwTransifexResult === 0)
            {
                $result['noBwTxTables'] = true;
            }

            // Check if cTransifex has an entry for a project
            $query->clear();
            $query->select('COUNT(' . $db->quoteName('id') . ')');
            $query->from($db->quoteName('#__ctransifex_projects'));

            $db->setQuery($query);

            $cTransifexResult = $db->loadResult();

            if ((int)$cTransifexResult > 0)
            {
                $result['cTxTablesExists'] = true;
            }
        }
        catch (Exception $e)
        {
            $result['cTxTablesExists'] = false;
        }

        // Check if artifacts of cTransifex exists
        // Language packages are stored in the package subfolder of media of the component. Each language project has its
        // own folder with subfolders for the languages. The folders of the projects are the only folders there, so if
        // there are folders present, there seems something to copy to exist,
        try
        {
            if (is_dir(Path::clean(JPATH_ROOT . '/media/com_ctransifex/packages')))
            {
                if (count(Folder::folders(JPATH_ROOT . '/media/com_ctransifex/packages')))
                {
                    $result['cTxArtifactsExists'] = true;
                }
            }
        }
        catch (UnexpectedValueException $e)
        {
            $result['error'] .= $e->getMessage();
        }

        return $result;
    }

    /**
     * Do the takeover component settings, tables and artifacts (language packages)
     *
     * The takeover of component settings has to be done first to get the organization name needed to adjust
     * project settings for tx
     *
     * @param   array  $takeoverCheck
     *
     * @return  void
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    private function doTakeover(array $takeoverCheck): void
    {
        // Import options of ctransifex to BwTransifex
        if ($takeoverCheck['cTxComponentExists'])
        {
            $organization = $this->takeoverComponentSettings();
            Factory::getApplication()->enqueueMessage(Text::_('COM_BWTRANSIFEX_TAKEOVER_COMPONENT_SETTINGS_SUCCESSFUL'), 'success');
        }
        else
        {
            $organization = '';
            Factory::getApplication()->enqueueMessage(Text::_('COM_BWTRANSIFEX_TAKEOVER_COMPONENT_OPTIONS_NOT_POSSIBLE'), 'warning');
        }

        // Import tables of ctransifex to BwTransifex
        $this->takeoverTables($organization);

        // Import artifacts of ctransifex to BwTransifex
        if ($takeoverCheck['cTxArtifactsExists'])
        {
            $this->takeoverArtifacts();
            Factory::getApplication()->enqueueMessage(Text::_('COM_BWTRANSIFEX_TAKEOVER_LANGUAGE_PACKAGES_SUCCESSFUL'), 'success');
        }
        else
        {
            Factory::getApplication()->enqueueMessage(Text::_('COM_BWTRANSIFEX_TAKEOVER_NO_ARTIFACTS'), 'warning');
        }

    }

    /**
     * Show the after installation/update screen
     *
     * @param boolean $update
     *
     * @return  void
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    public function showFinished(bool $update): void
    {
        $lang = Factory::getApplication()->getLanguage();

        // First load english files
        $lang->load('com_bwtransifex.sys', JPATH_ADMINISTRATOR, 'en_GB', true);
        $lang->load('com_bwtransifex', JPATH_ADMINISTRATOR, 'en_GB', true);

        // The load current language
        $lang->load('com_bwtransifex.sys', JPATH_ADMINISTRATOR, null, true);
        $lang->load('com_bwtransifex', JPATH_ADMINISTRATOR, null, true);

        $show_update = false;
        $show_right  = false;
        $lang_ver    = substr($lang->getTag(), 0, 2);

        if ($lang_ver != 'de')
        {
            $forum    = "https://www.boldt-webservice.de/en/forum-en/forum/bwtransifex.html";
            $manual = "https://www.boldt-webservice.de/index.php/en/forum-en/manuals/bwtransifex-manual.html";
        }
        else
        {
            $forum = "https://www.boldt-webservice.de/de/forum/bwtransifex.html";
            $manual = "https://www.boldt-webservice.de/index.php/de/forum/handb%C3%BCcher/handbuch-zu-bwtransifex.html";
        }

        if ($update)
        {
            $string_special = Text::_('COM_BWTRANSIFEX_INSTALLATION_UPDATE_SPECIAL_NOTE_DESC');
        }
        else
        {
            $string_special = Text::_('COM_BWTRANSIFEX_INSTALLATION_INSTALLATION_SPECIAL_NOTE_DESC');
        }

        $string_new         = Text::_('COM_BWTRANSIFEX_INSTALLATION_UPDATE_NEW_DESC');
        $string_improvement = Text::_('COM_BWTRANSIFEX_INSTALLATION_UPDATE_IMPROVEMENT_DESC');
        $string_bugfix      = Text::_('COM_BWTRANSIFEX_INSTALLATION_UPDATE_BUGFIX_DESC');

        if (($string_bugfix != '' || $string_improvement != '' || $string_new != '') && $update)
        {
            $show_update = true;
        }

        if ($show_update || $string_special != '')
        {
            $show_right = true;
        }

        $asset_path = 'media/com_bwtransifex';
        $image_path = Uri::root() . 'media/com_bwtransifex/images';
        $css_file = '/css/install.css';

        if (version_compare(JVERSION, '4.0.0', 'lt'))
        {
            $css_file = '/css/install_J3.css';
        }
        ?>

        <link rel="stylesheet" href="<?php echo Uri::root() . $asset_path . $css_file; ?>" type="text/css" />

        <div id="bw_installation_header" class="text-center">
            <a href="https://www.boldt-webservice.de" target="_blank">
                <img class="img-fluid border-0" data-test="install-head-image" src="<?php echo Uri::root() . $asset_path . '/images/bw_header.png'; ?>" alt="Boldt Webservice" />
            </a>
        </div>
        <div class="top_line"></div>

        <div id="bw_installation_outer" class="row">
            <div class="col-lg-12 text-center p-2 mt-2">
                <h1 data-test="install-header"><?php echo Text::_('COM_BWTRANSIFEX_INSTALLATION_WELCOME') ?></h1>
            </div>
            <div id="bw_installation_left" class="col-lg-6 mb-2">
                <div class="bw_installation_welcome">
                    <p data-test="install-description"><?php echo Text::_('COM_BWTRANSIFEX_DESCRIPTION') ?></p>
                </div>
                <div class="bw_installation_finished text-center">
                    <h2 data-test="install-success-text">
                        <?php
                        if ($update)
                        {
                            echo Text::sprintf('COM_BWTRANSIFEX_UPGRADE_SUCCESSFUL', $this->release);
                        }
                        else
                        {
                            echo Text::sprintf('COM_BWTRANSIFEX_INSTALLATION_SUCCESSFUL', $this->release);
                        }
                        ?>
                    </h2>
                </div>
                <?php
                if ($show_right)
                { ?>
                    <div class="cpanel text-center mb-3">
                        <div class="icon" >
                            <a href="<?php echo Route::_('index.php?option=com_bwtransifex&view=projects'); ?>">
                                <?php echo HtmlHelper::_(
                                    'image',
                                    $image_path . '/icon-48-bwtransifex.png',
                                    Text::_('COM_BWTRANSIFEX_INSTALLATION_GO_BWTRANSIFEX'),
                                    'data-test="install-goto-icon"'
                                ); ?>
                                <span data-test="install-goto-text"><?php echo Text::_('COM_BWTRANSIFEX_INSTALLATION_GO_BWTRANSIFEX'); ?></span>
                            </a>
                        </div>
                        <div class="icon">
                            <a href="<?php echo $manual; ?>" target="_blank" data-test="install-manual-link">
                                <?php echo HtmlHelper::_(
                                    'image',
                                    $image_path . '/icon-48-manual.png',
                                    Text::_('COM_BWTRANSIFEX_INSTALLATION_MANUAL'),
                                    'data-test="install-manual-icon"'
                                ); ?>
                                <span data-test="install-manual-text"><?php echo Text::_('COM_BWTRANSIFEX_INSTALLATION_MANUAL'); ?></span>
                            </a>
                        </div>
                        <div class="icon">
                            <a href="<?php echo $forum; ?>" target="_blank" data-test="install-forum-link">
                                <?php echo HtmlHelper::_(
                                    'image',
                                    $image_path . '/icon-48-forum.png',
                                    Text::_('COM_BWTRANSIFEX_INSTALLATION_FORUM'),
                                    'data-test="install-forum-icon"'
                                ); ?>
                                <span data-test="install-forum-text"><?php echo Text::_('COM_BWTRANSIFEX_INSTALLATION_FORUM'); ?></span>
                            </a>
                        </div>
                    </div>
                    <?php
                } ?>
            </div>

            <div id="bw_installation_right" class="col-lg-6">
                <?php
                if ($show_right)
                {
                    if ($string_special != '')
                    { ?>
                        <div class="bw_installation_specialnote">
                            <h2 data-test="install-special-header"><?php echo Text::_('COM_BWTRANSIFEX_INSTALLATION_SPECIAL_NOTE_LBL') ?></h2>
                            <p class="urgent" data-test="install-special-text"><?php echo $string_special; ?></p>
                        </div>
                        <?php
                    } ?>

                    <?php
                    if ($show_update)
                    { ?>
                        <div class="bw_installation_updateinfo mb-3 p-3">
                            <h2 class="mb-3"><?php echo Text::_('COM_BWTRANSIFEX_INSTALLATION_UPDATEINFO') ?></h2>
                            <?php echo Text::_('COM_BWTRANSIFEX_INSTALLATION_CHANGELOG_INFO'); ?>
                            <?php if ($string_new != '')
                            { ?>
                                <h3 class="mb-2" data-test="update-header-new"><?php echo Text::_('COM_BWTRANSIFEX_INSTALLATION_UPDATE_NEW_LBL') ?></h3>
                                <p data-test="update-text-new"><?php echo $string_new; ?></p>
                            <?php
                            } ?>
                            <?php if ($string_improvement != '')
                            { ?>
                                <h3 class="mb-2" data-test="update-header-improvement"><?php echo Text::_('COM_BWTRANSIFEX_INSTALLATION_UPDATE_IMPROVEMENT_LBL') ?></h3>
                                <p data-test="update-text-improvement"><?php echo $string_improvement; ?></p>
                            <?php
                            } ?>
                            <?php if ($string_bugfix != '')
                            { ?>
                                <h3 class="mb-2" data-test="update-header-bugfix"><?php echo Text::_('COM_BWTRANSIFEX_INSTALLATION_UPDATE_BUGFIX_LBL') ?></h3>
                                <p data-test="update-text-bugfix"><?php echo $string_bugfix; ?></p>
                            <?php
                            } ?>
                        </div>
                        <?php
                    }
                }
                else
                { ?>
                    <div class="cpanel text-center mb-3">
                        <div class="icon" >
                            <a href="<?php echo Route::_('index.php?option=com_bwtransifex&view=projects'); ?>">
                                <?php echo HtmlHelper::_(
                                    'image',
                                    $image_path . '/icon-48-bwtransifex.png',
                                    Text::_('COM_BWTRANSIFEX_INSTALLATION_GO_BWTRANSIFEX'),
                                    'data-test="install-goto-icon"'
                                ); ?>
                                <span data-test="install-goto-text"><?php echo Text::_('COM_BWTRANSIFEX_INSTALLATION_GO_BWTRANSIFEX'); ?></span>
                            </a>
                        </div>
                        <div class="icon">
                            <a href="<?php echo $manual; ?>" target="_blank" data-test="install-manual-link">
                                <?php echo HtmlHelper::_(
                                    'image',
                                    $image_path . '/icon-48-manual.png',
                                    Text::_('COM_BWTRANSIFEX_INSTALLATION_MANUAL'),
                                    'data-test="install-manual-icon"'
                                ); ?>
                                <span data-test="install-manual-text"><?php echo Text::_('COM_BWTRANSIFEX_INSTALLATION_MANUAL'); ?></span>
                            </a>
                        </div>
                        <div class="icon">
                            <a href="<?php echo $forum; ?>" target="_blank" data-test="install-forum-link">
                                <?php echo HtmlHelper::_(
                                    'image',
                                    $image_path . '/icon-48-forum.png',
                                    Text::_('COM_BWTRANSIFEX_INSTALLATION_FORUM'),
                                    'data-test="install-forum-icon"'
                                ); ?>
                                <span data-test="install-forum-text"><?php echo Text::_('COM_BWTRANSIFEX_INSTALLATION_FORUM'); ?></span>
                            </a>
                        </div>
                    </div>
                    <?php
                } ?>
            </div>
            <div class="clr clearfix"></div>

            <div class="bw_installation_footer col-12 text-center my-3">
                <p class="small" data-test="install-copyright-text">
                    <?php echo Text::_('&copy; ');
                    echo date(" Y") ?> by
                    <a href="https://www.boldt-webservice.de" target="_blank" data-test="install-copyright-link">Boldt Webservice</a>
                </p>
            </div>
        </div>
    <?php
    }

    /**
     * import the settings of cTransifex to BwTransifex
     *
     * @return  string the name of the organization recorded at transifex
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    protected function takeoverComponentSettings(): string
    {
        $db  = $this->getDb();
        $this->db = $db;
        $query = $db->getQuery(true);

        // ctransifex exists?
        $query->select($db->quoteName('params'));
        $query->from($db->quoteName('#__extensions'));
        $query->where($db->quoteName('element') . " LIKE '%com_ctransifex%'");

        $db->setQuery($query);

        $cTransifexParams = $db->loadResult();
        $this->setComponentParams($cTransifexParams);


        $cParams = new Registry($cTransifexParams);
        return $cParams->get('tx_username', '');
    }

    /**
     * Convert created time column and set appropriate null date if needed
     *
     * @param array $currentTableEntry
     *
     * @return array
     *
     * @since 1.0.0
     */
    protected function adjustCreatedTimeColumn(array $currentTableEntry): array
    {
        if (key_exists('created', $currentTableEntry))
        {
            $currentTableEntry['created_time'] = $currentTableEntry['created'];
            unset($currentTableEntry['created']);

            if ($currentTableEntry['created_time'] == 0 || $currentTableEntry['created_time'] === $this->db->getNullDate())
            {
                $currentTableEntry['created_time'] = Factory::getDate()->toSql();
            }
        }

        return $currentTableEntry;
    }

    /**
     * Convert modified column and set appropriate null date if needed
     *
     * @param array $currentTableEntry
     *
     * @return array
     *
     * @since 1.0.0
     */
    protected function adjustModifiedTimeColumn(array $currentTableEntry): array
    {
        if (key_exists('modified', $currentTableEntry))
        {
            $currentTableEntry['modified_time'] = $currentTableEntry['modified'];
            unset($currentTableEntry['modified']);

            if ($currentTableEntry['modified_time'] == 0 || $currentTableEntry['modified_time'] === $this->db->getNullDate())
            {
                unset($currentTableEntry['modified_time']);
            }
        }

        return $currentTableEntry;
    }

    /**
     * Convert state column
     *
     * @param array $currentTableEntry
     *
     * @return array
     *
     * @since 1.0.0
     */
    protected function adjustStateColumn(array $currentTableEntry): array
    {
        if (key_exists('state', $currentTableEntry))
        {
            $currentTableEntry['published'] = $currentTableEntry['state'];
            unset($currentTableEntry['state']);
        }

        return $currentTableEntry;
    }

    /**
     * Convert completed column
     *
     * @return void
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    protected function adjustCompletedColumn(): void
    {
        try
        {
            // Modify completed from int to float
            $db  = $this->getDb();

            $query = "ALTER TABLE {$db->quoteName('#__bwtransifex_languages')} MODIFY {$db->quoteName('completed')} float(5,2) NOT NULL";

            $db->setQuery($query);
            $db->execute();

            $query = "ALTER TABLE {$db->quoteName('#__bwtransifex_zips')} MODIFY {$db->quoteName('completed')} float(5,2) NOT NULL";

            $db->setQuery($query);
            $db->execute();
        }
        catch (RuntimeException $e)
        {
            $message = Text::_('COM_BWTRANSIFEX_TAKEOVER_TABLES_ERROR');
            $message .= $e->getMessage();
            Factory::getApplication()->enqueueMessage($message, 'warning');
        }
    }

    /**
     * Convert old tx config settings section names to new ones
     *
     * @param array  $currentTableEntry
     * @param string $organization
     *
     * @return array
     *
     * @since 1.0.0
     */
    protected function adjustProjectTxSettings(array $currentTableEntry, string $organization) :array
    {
        $newConfigEntries    = array();

        // Get project tx config
        $configEntries = parse_ini_string($currentTableEntry['transifex_config'], true);

        // Create new config array with new tx requirements on resource names
        foreach ($configEntries as $configKey => $configValues)
        {
            // For section main nothing has changed
            if ($configKey === 'main')
            {
                $newConfigEntries[$configKey] = $configValues;
            }
            else
            {
                // If section names for resources are old ones, convert them
                if ((stripos($configKey, 'o:') === false) || (stripos($configKey, 'p:') === false) || (stripos($configKey, 'o:') === false))
                {
                    // Get resource name
                    $oldKeyParts = explode('.', $configKey);
                    $resource    = end($oldKeyParts);

                    // Build new section name
                    $newKey  = 'o:' . $organization;
                    $newKey .= ':p:' . $currentTableEntry['transifex_slug'];
                    $newKey .= ':r:' . $resource;

                    $newConfigEntries[$newKey] = $configValues;
                }
                // If section name is new already, do nothing
                else
                {
                    $newConfigEntries[$configKey] = $configValues;
                }
            }
        }

        // Convert new config array to ini string
        $iniString = '';

        foreach ($newConfigEntries as $section => $values)
        {
            $iniString .= "[$section]\n";

            foreach ($values as $key => $value)
            {
                $iniString .= "$key = \"$value\"\n";
            }
        }

        $currentTableEntry['transifex_config'] = $iniString;

        // Add new parameter 'update_always'
        $params = json_decode($currentTableEntry['params'], true);
        $params['update_always'] = "0";
        $currentTableEntry['params'] = json_encode($params);

        return $currentTableEntry;
    }

    /**
     * Take over tables
     * - Copy the tables from com_ctransifex to com_bw_transifex
     * - Do adjustments to columns definition
     * - Do adjustments to settings for transifex.com if possible
     *
     * If com_ctransifex is not installed, parameter organization is an empty string, because it could not be determined.
     * So the settings for transifex.com could not be adjusted.
     *
     * @param   string  $organization
     *
     * @return void
     *
     * @throws Exception
     *
     * @since 1,0,0
     */
    private function takeoverTables(string $organization): void
    {
        $db  = $this->getDb();
        $this->db = $db;
        $query = $db->getQuery(true);

        try
        {
            // Import tables of ctransifex to BwTransifex
            $neededTables = array(
                'languages',
                'projects',
                'resources',
                'zips',
            );

            foreach ($neededTables as $neededTable)
            {
                $currentSrcTable  = '#__ctransifex_' . $neededTable;
                $currentDestTable = '#__bwtransifex_' . $neededTable;

                $query->clear();
                $query->select('*');
                $query->from($db->quoteName($currentSrcTable));

                $db->setQuery($query);

                $currentTableEntries = $db->loadAssocList();

                $query->clear();

                foreach ($currentTableEntries as $currentTableEntry)
                {
                    $query->clear();

                    $currentTableEntry = $this->adjustCreatedTimeColumn($currentTableEntry);
                    $currentTableEntry = $this->adjustModifiedTimeColumn($currentTableEntry);
                    $currentTableEntry = $this->adjustStateColumn($currentTableEntry);

                    if ($neededTable === 'projects')
                    {
                        if ($organization !== '')
                        {
                            $currentTableEntry = $this->adjustProjectTxSettings($currentTableEntry, $organization);
                        }
                        else
                        {
                            Factory::getApplication()->enqueueMessage(Text::_('COM_BWTRANSIFEX_TAKEOVER_TX_SETTINGS_NOT_POSSIBLE'), 'warning');
                        }
                    }

                    $columns = array_keys($currentTableEntry);
                    $values  = array_values($currentTableEntry);

                    $sanitizedColumns = array();
                    $sanitizedValues  = array();

                    // Sanitize columns
                    foreach ($columns as $column)
                    {
                        $sanitizedColumns[] = $db->quoteName($column);
                    }

                    // Sanitize values
                    foreach ($values as $value)
                    {
                        $sanitizedValues[] = $db->quote($value);
                    }

                    $query->insert($db->quoteName($currentDestTable));
                    $query->columns($sanitizedColumns);
                    $query->values(implode(',', $sanitizedValues));

                    $db->setQuery($query);
                    $db->execute();
                }
            }

            $this->adjustCompletedColumn();

            Factory::getApplication()->enqueueMessage(Text::_('COM_BWTRANSIFEX_TAKEOVER_TABLES_SUCCESSFUL'), 'success');
        }
        catch (RuntimeException $e)
        {
            $message = Text::_('COM_BWTRANSIFEX_TAKEOVER_TABLES_ERROR');
            $message .= $e->getMessage();
            Factory::getApplication()->enqueueMessage($message, 'warning');
        }
    }

    /**
     * Copy the artifacts from com_ctransifex to com_bw_transifex
     *
     * - copy common files based in package folder
     * - create new project folders and their language folders
     * - copy language files
     *
     * The tmp folder of each language is ignored
     *
     * @return void
     *
     * @throws Exception
     *
     * @since 1,0,0
     */
    private function takeoverArtifacts(): void
    {
        $srcBase  = JPATH_ROOT . '/media/com_ctransifex/packages';
        $destBase = JPATH_ROOT . '/media/com_bwtransifex/packages/';

        try
        {
            // Get common files
            $commonFiles = Folder::files($srcBase, '.', false);

            // Copy common files
            foreach ($commonFiles as $file)
            {
                if (!is_file(Path::clean($destBase . $file)))
                {
                    File::copy($srcBase . '/' . $file, $destBase . $file);
                }
            }

            // Get project folders
            $projectFolders = Folder::folders($srcBase);

            foreach ($projectFolders as $folder)
            {
                // Create projects folders
                if (!is_dir(Path::clean($destBase . $folder)))
                {
                    Folder::create($destBase . $folder);
                }

                // Get project language folders
                $projectSubFolders = Folder::folders($srcBase . '/' . $folder);

                foreach ($projectSubFolders as $subFolder)
                {
                    // Create project language folders
                    if (!is_dir(Path::clean($destBase . $folder . '/' . $subFolder)))
                    {
                        Folder::create($destBase . $folder . '/' . $subFolder);
                    }

                    // Get project language files
                    $subFolderFiles = Folder::files($srcBase . '/' . $folder . '/' . $subFolder);

                    // Copy project language files
                    foreach ($subFolderFiles as $file) {
                        if (!is_file(Path::clean($destBase . $folder . '/' . $subFolder . '/' . $file)))
                        {
                            File::copy($srcBase . '/' . $folder . '/' . $subFolder . '/' . $file, $destBase . $folder . '/' . $subFolder . '/' . $file);
                        }
                    }
                }
            }
        }
        catch (UnexpectedValueException $e)
        {
            $message = Text::_('COM_BWTRANSIFEX_TAKEOVER_ARTIFACTS_ERROR');
            $message .= $e->getMessage();
            Factory::getApplication()->enqueueMessage($message, 'warning');
        }
    }
}
