<?php
/**
 * BwTransifex Component
 *
 * BwTransifex package helper class for the component backend
 *
 * BwTransifex is a largely reworked fork of Jonathan Daniel Dimitrov´s cTransifex package
 *
 * @version 1.0.1
 * @package BwTransifex
 * @subpackage BwTransifex Component Admin
 * @author Romana Boldt
 * @copyright (C) 2025 Boldt Webservice <forum@boldt-webservice.de>
 * @support https://www.boldt-webservice.de/en/forum-en/forum/bwtransifex.html
 * @license GNU/GPL, see LICENSE.txt
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

namespace BoldtWebservice\Component\BwTransifex\Administrator\Helper;

defined('_JEXEC') or die('Restricted access');

use DOMDocument;
use Exception;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\Filesystem\File;
use Joomla\Filesystem\Folder;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Log\Log;
use Joomla\CMS\Uri\Uri;
use Joomla\Registry\Registry;
use RecursiveDirectoryIterator;
use RecursiveIteratorIterator;
use RuntimeException;
use SimpleXMLElement;
use ZipArchive;

/**
 * Define the BwTransifex package helper class
 *
 * @package BwTransifex Admin
 *
 * @since   1.0.0
 */
class BwTransifexHelperPackage
{
    /**
     * Saves a language file
     *
     * @param array  $file     the file data
     * @param string $jLang    joomla language
     * @param object $project  the project object
     * @param string $resource the resource name
     * @param array  $config   the config object
     *
     * @return bool
     *
     * @throws Exception
     * @since 1.0.0
     */
    public static function saveLangFile(array $file, string $jLang, object $project, string $resource, array $config): bool
    {
        // Find out the fileName and his location
        $orgSlug      = ComponentHelper::getParams('com_bwtransifex')->get('tx_username');
        $projectSlug  = "o:" . $orgSlug . ":p:" . $project->transifex_slug . ":r:". $resource;
        // Search pattern is / (or \ for old tx api)
        $fileFilter   = explode('/', $config[$projectSlug]['file_filter']);
        $fileName     = ltrim(str_replace('<lang>', $jLang, end($fileFilter)), '.');
        $isAdmin      = false;
        $isInstall    = false;
        $adminPath    = JPATH_ROOT . '/media/com_bwtransifex/packages/' . $project->transifex_slug . '/' . $jLang . '/tmp/admin/';
        $frontendPath = JPATH_ROOT . '/media/com_bwtransifex/packages/' . $project->transifex_slug . '/' . $jLang . '/tmp/frontend/';
        $installPath  = JPATH_ROOT . '/media/com_bwtransifex/packages/' . $project->transifex_slug . '/' . $jLang . '/tmp/installation/';
        $path         = $frontendPath . $fileName;
        $params       = new Registry($project->params);

        // Determine location by file filter
        if ($params->get('determine_location', 1))
        {
            if (in_array('admin', $fileFilter) || in_array('administrator', $fileFilter) || in_array('backend',
                    $fileFilter))
            {
                $isAdmin = true;
            }

            if (in_array('install', $fileFilter) || in_array('installation', $fileFilter))
            {
                $isInstall = true;
            }
        }
        // Determine location by resource name
        else
        {
            if (strstr($resource, 'admin') || strstr($resource, 'administrator') || strstr($resource, 'backend'))
            {
                $isAdmin = true;
            }

            if (strstr($resource, 'install') || strstr($resource, 'installation'))
            {
                $isInstall = true;
            }
        }

        if ($isAdmin)
        {
            $path = $adminPath . $fileName;
        }

        if ($isInstall)
        {
            $path = $installPath . $fileName;
        }

        $newFileData = self::replaceVariables($file['data'], $project, $jLang);

        return File::write($path, $newFileData);
    }

    /**
     * Create a language pack
     *
     * @param string $jLang   - the joomla language
     * @param object $project - the project object
     *
     * @return array
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    public static function package(string $jLang, object $project): array
    {
        // @ToDo: Move to before getting files
        $updateNeeded     = Factory::getApplication()->getSession()->get('updateNeeded', false, 'bwtransifex');
        $response         = array();

        // If there has nothing changed, we are ready
        if (!$updateNeeded)
        {
            $response['message'] = Text::_('COM_BWTRANSIFEX_LANGPACK_NO_CHANGE') . $jLang;
            $response['status'] = 'success';
            $response['action'] = 'nothing';

            return $response;
        }

        // If there are new translations, get new version for language package
        $languagePackVersion = self::getNewLanguagePackageVersion($project->id, $jLang);

        $folder    = JPATH_ROOT . '/media/com_bwtransifex/packages/' . $project->transifex_slug . '/' . $jLang;
        $tmpFolder = JPATH_ROOT . '/media/com_bwtransifex/packages/' . $project->transifex_slug . '/' . $jLang . '/tmp';

        // Make sure tmp folder as working folder exists
        if (!is_dir($tmpFolder))
        {
            Folder::create($tmpFolder);
        }

        $projectName = str_replace(' ', '_', $project->extension_name);
        $rawFileName = $jLang . '.' . $projectName . '.' . $languagePackVersion . '.zip';
        $zipFile     = $folder . '/' . $rawFileName;
        $tmpZipFile  = $tmpFolder . '/' . $rawFileName;

        // Make sure that we are always creating a new zip file
        if (is_file($tmpZipFile))
        {
            File::delete($tmpZipFile);
        }

        // Create installation manifest
        if (!self::generateInstallXML($tmpFolder, $jLang, $project))
        {
            $response['message'] = 'Error creating install manifest for ' . $jLang;
            $response['status'] = 'error';

            return $response;
        }

        // Pack language package
        $zipResult = self::zip($tmpFolder, $tmpZipFile);

        if ($zipResult['status'] == 'success')
        {
            // Clean up
            File::copy($tmpZipFile, $zipFile);

            if (is_dir($tmpFolder . '/admin'))
            {
                Folder::delete($tmpFolder . '/admin');
            }

            if (is_dir($tmpFolder . '/frontend'))
            {
                Folder::delete($tmpFolder . '/frontend');
            }

            if (is_file($tmpFolder . '/' . $projectName . '_' . $jLang . '.xml'))
            {
                File::delete($tmpFolder . '/' . $projectName . '_' . $jLang . '.xml');
            }

            if (is_file($tmpZipFile))
            {
                File::delete($tmpZipFile);
            }

            // Create new definition for update server
            if (!self::updateUpdateServerXML($jLang, $project))
            {
                $response['message'] = 'Error creating update server manifest for ' . $jLang;
                $response['status'] = 'error';

                return $response;
            }

            // Create summary for update servers
            if (!self::createTranslationXML($jLang, $project))
            {
                $response['message'] = 'Error creating/updating translation list for ' . $jLang;
                $response['status'] = 'error';

                return $response;
            }

            $response['message'] = Text::_("COM_BWTRANSIFEX_LANGPACK_CREATION_DONE") . $jLang;
            $response['status'] = 'success';
            $response['action'] = 'changed';

            return $response;
        }

        $response['message'] = 'Error creating/updating translation list for ' . $jLang . ': <b>' . $zipResult['message'] . '</b>';
        $response['status'] = 'error';

        return $response;
    }

    /**
     * Recursively go through each file in the folder and ad it to the archive.
     *
     * @param string $source      path to the source folder
     * @param string $destination path to the destination folder
     *
     * @return array
     *
     * @since 1.0.0
     */
    private static function zip(string $source, string $destination): array
    {
        $result = array(
            'status' => 'error',
        );

        if (!extension_loaded('zip'))
        {
            $result['message'] = 'Php zip module is not installed on this webserver';

            return $result;
        }

        if (!file_exists($source))
        {
            $result['message'] = $source . ' not found';

            return $result;
        }

        $zip = new ZipArchive;

        if (!$zip->open($destination, ZIPARCHIVE::CREATE))
        {
            $result['message'] = 'Could not create zip archive ' . $destination;

            return $result;
        }

        $source = str_replace('\\', '/', realpath($source));

        if (is_dir($source) === true)
        {
            $iterator = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($source));

            foreach ($iterator as $key => $value)
            {
                if (!is_dir($key))
                {
                    Log::addLogger(array('text_file' => 'com_bwtransifex.error.php'));
                    Log::add('Zip realpath: ' . realpath($key));
                    Log::add('Zip entry name: ' . substr($key, strlen($source) + 1));
                    if (!$zip->addFile(realpath($key), substr($key, strlen($source) + 1)))
                    {
                        $result['message'] = 'Could not add file ' . $key . ' to ' . $source;

                        return $result;
                    }
                }
            }
        }

        if ($zip->close())
        {
            $result['status'] = 'success';
            $result['message'] = 'Zip archive created successfully';
        }
        else
        {
            $result['message'] = 'Could not close created zip archive';
        }

        return $result;
    }

    /**
     * Generate the installer manifest
     *
     * @param string $folder  the folder path
     * @param string $jLang   the language
     * @param object $project the project object
     *
     * @return bool
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    private static function generateInstallXml(string $folder, string $jLang, object $project): bool
    {
        $mediaXML = JPATH_ROOT . '/media/com_bwtransifex/packages/install_' . $project->transifex_slug . '.xml';

        if (file_exists($mediaXML))
        {
            $dummyXml = file_get_contents($mediaXML);
        }
        else
        {
            $dummyXml = file_get_contents(JPATH_ADMINISTRATOR . '/components/com_bwtransifex/assets/install.xmt');
            File::write(JPATH_ROOT . '/media/com_bwtransifex/packages/install_' . $project->transifex_slug . '.xml', $dummyXml);
        }

        $params          = ComponentHelper::getParams('com_bwtransifex');
        $adminFiles      = '';
        $frontendFiles   = '';
        $iso             = explode('-', $jLang);
        $isoLangName     = ucfirst(BwTransifexHelperLanguage::code2ToName($iso[0]));
        $isoCountryName  = BwTransifexHelperLanguage::code2ToCountry($iso[1]);
        $isoLanguage     = "$isoLangName ($isoCountryName)";
        $languageVersion = self::getNewLanguagePackageVersion($project->id, $jLang);
        $projectName     = str_replace(' ', '_', $project->extension_name);
        $rawServerPath   = substr(self::cleanupPath($project->params['updateserver_path']), 1);
        $updateServerUrl = Uri::root() . $rawServerPath . 'translationlist.xml';

        $content = str_replace('@@EXTENSION_NAME@@', $project->extension_name, $dummyXml);
        $content = str_replace('@@VERSION@@', $languageVersion, $content);
        $content = str_replace('@@CREATION_DATE@@', date('d.m.Y'), $content);
        $content = str_replace('@@AUTHOR@@', $params->get('author'), $content);
        $content = str_replace('@@AUTHOR_EMAIL@@', $params->get('author_email'), $content);
        $content = str_replace('@@AUTHOR_URL@@', $params->get('author_url'), $content);
        $content = str_replace('@@COPYRIGHT@@', str_replace('@@YEAR@@', date('Y'), $params->get('copyright')),
            $content);
        $content = str_replace('@@DESCRIPTION@@', Text::_('COM_BWTRANSIFEX_INSTALL_XML_DESCRIPTION'), $content);
        $content = str_replace('@@LANGUAGE@@', $isoLanguage, $content);
        $content = str_replace('@@UPDATE_SERVER@@', $updateServerUrl, $content);

        $admin = self::getFiles($folder . '/admin');

        if ($admin)
        {
            $adminFiles = '<files folder="admin" target="administrator/language/' . $jLang . '">' . "\n" . $admin . "\n" . '        </files>';
        }

        $content  = str_replace('@@ADMIN_FILENAMES@@', $adminFiles, $content);

        $frontend = self::getFiles($folder . '/frontend');

        if ($frontend)
        {
            $frontendFiles = '<files folder="frontend" target="language/' . $jLang . '">' . "\n" . $frontend . "\n" . '        </files>';
        }

        $content = str_replace('@@FRONTEND_FILENAMES@@', $frontendFiles, $content);

        if (File::write($folder . '/' . $projectName . '_' . $jLang . '.xml', $content))
        {
            return true;
        }

        return false;
    }

    /**
     * Update the update server file or create it if it does not exist
     *
     * @param string $jLang   the language
     * @param object $project the project object
     *
     * @return bool
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    private static function updateUpdateServerXML(string $jLang, object $project): bool
    {
        $updateServerPath = JPATH_ROOT . self::cleanupPath($project->params['updateserver_path']);
        $projectName      = str_replace(' ', '_', $project->extension_name);
        $packagesPath     = JPATH_ROOT . '/media/com_bwtransifex/packages';

        // Get used update server file
        $updateServerFile = $updateServerPath . '/' . $jLang . '.' . $projectName . '.xml';

        if (file_exists($updateServerFile))
        {
            $currentUpdateServerXML = file_get_contents($updateServerFile);
        }
        else
        {
            $currentUpdateServerXML = file_get_contents(JPATH_ADMINISTRATOR . '/components/com_bwtransifex/assets/updateserver_base.xmt');
        }

        // Get update server template
        $templateXmlFile = $packagesPath . '/updateserver_' . $project->transifex_slug . '.xmt';

        if (file_exists($templateXmlFile))
        {
            $templateXml = file_get_contents($templateXmlFile);
        }
        else
        {
            $templateXml = file_get_contents(JPATH_ADMINISTRATOR . '/components/com_bwtransifex/assets/updateserver_tpl.xmt');
            if (!File::write($templateXmlFile, $templateXml))
            {
                Log::addLogger(array('text_file' => 'com_bwtransifex.error.php'));
                Log::add('Something went wrong when we tried to get the update server manifest template.'. Log::ERROR);

                return false;
            }
        }

        // Fill update server template
        $languagePackVersion = self::getNewLanguagePackageVersion($project->id, $jLang);
        $zipUrl              = Uri::root() . 'media/com_bwtransifex/packages/' . $project->transifex_slug . '/' . $jLang . '/' . $jLang . '.' . $projectName . '.' . $languagePackVersion . '.zip';

        $iso             = explode('-', $jLang);
        $isoLangName     = ucfirst(BwTransifexHelperLanguage::code2ToName($iso[0]));
        $isoCountryName  = BwTransifexHelperLanguage::code2ToCountry($iso[1]);
        $isoLanguage     = "$isoLangName ($isoCountryName)";
        $languageVersion = self::getNewLanguagePackageVersion($project->id, $jLang);
        $element         = $projectName . '_' . $jLang;

        $addContent = str_replace('@@EXTENSION_NAME@@', $project->extension_name, $templateXml);
        $addContent = str_replace('@@VERSION@@', $languageVersion, $addContent);
        $addContent = str_replace('@@ELEMENT@@', $element, $addContent);
        $addContent = str_replace('@@ZIPURL@@', $zipUrl, $addContent);
        $addContent = str_replace('@@LANGUAGE@@', $isoLanguage, $addContent);

        // Add filled update server template to update server file
        $newUpdateServerXML = str_replace('</updates>', $addContent, $currentUpdateServerXML);
        $newUpdateServerXML .= "</updates>";

        if (File::write($updateServerFile, $newUpdateServerXML))
        {
            return true;
        }

        return false;
    }

    /**
     * Method to update translation list XML or create it if it does  not exist
     *
     * @param string $jLang   the language
     * @param object $project the project object
     *
     * @return bool
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    private static function createTranslationXML(string $jLang, object $project): bool
    {
        $extensionFound   = false;
        $projectName      = str_replace(' ', '_', $project->extension_name);
        $updateServerPath = JPATH_ROOT . self::cleanupPath($project->params['updateserver_path']);
        $rawServerPath    = substr(self::cleanupPath($project->params['updateserver_path']), 1);
        $updateServerUrl  = Uri::root() . $rawServerPath . $jLang . '.' . $projectName . '.xml';

        // Get used translation list file
        $translationListFile = $updateServerPath . '/translationlist.xml';
        $languageVersion     = self::getNewLanguagePackageVersion($project->id, $jLang);
        $element             = $projectName . '_' . $jLang;

        if (file_exists($translationListFile))
        {
            $currentTranslationListXML = simplexml_load_file($translationListFile);

            foreach ($currentTranslationListXML->extension as $extension)
            {
                if ($extension['element'] == $element)
                {
                    $extensionFound          = true;
                    $extension['version']    = $languageVersion;
                    $extension['detailsurl'] = $updateServerUrl;
                    $extension['element']    = $element;
                }
            }
        }
        else
        {
            $currentTranslationListXML = new SimpleXMLElement('<?xml version="1.0" encoding="utf-8"?><extensionset></extensionset>');
            $currentTranslationListXML->addAttribute('name', $project->title);
            $currentTranslationListXML->addAttribute('description', 'List of all language pack update servers');
        }

        if (!$extensionFound)
        {
            $newExtension = $currentTranslationListXML->addChild('extension');
            $newExtension->addAttribute('name', $jLang . ' language for ' . $project->extension_name);
            $newExtension->addAttribute('element', $element);
            $newExtension->addAttribute('type', 'file');
            $newExtension->addAttribute('version', $languageVersion);
            $newExtension->addAttribute('detailsurl', $updateServerUrl);
        }

        $dom                     = new DOMDocument('<?xml version="1.0" encoding="utf-8"?>');
        $dom->preserveWhiteSpace = false;
        $dom->formatOutput       = true;
        $dom->loadXML($currentTranslationListXML->asXML());
        $newContent = $dom->saveXML();

        if (File::write($translationListFile, $newContent))
        {
            return true;
        }

        return false;
    }

    /**
     * Get the files of a specific folder
     *
     * @param string $folder - the path to the folder
     *
     * @return string
     *
     * @since 1.0.0
     */
    private static function getFiles(string $folder): string
    {
        $files = Folder::files($folder);
        $xml   = array();

        if (is_array($files))
        {
            foreach ($files as $file)
            {
                $xml[] = '            <filename>' . $file . '</filename>';
            }
        }

        return implode("\n", $xml);
    }

    /**
     * Get the new version of a language package for a given project
     *
     * @param integer $projectId The id of the project
     * @param string  $language  The language to look for
     *
     * @return string
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    public static function getNewLanguagePackageVersion(int $projectId, string $language): string
    {
        $projectModel = Factory::getApplication()
            ->bootComponent('com_bwtransifex')
            ->getMVCFactory()
            ->createModel('Project', 'Administrator');
        $project      = $projectModel->getItem($projectId);

        $projectVersion  = $project->params['project_version'];
        $langPackVersion = self::getCurrentLanguagePackageVersion($projectId, $language);

        if ($langPackVersion === '')
        {
            return $projectVersion . '.1';
        }

        $versionArray   = explode('.', $langPackVersion);
        $langSubVersion = array_pop($versionArray);

        $langPackMainVersion = implode('.', $versionArray);

        if ($projectVersion === $langPackMainVersion)
        {
            $newVersion = $projectVersion . '.' . ($langSubVersion + 1);
        }
        else
        {
            $newVersion = $projectVersion . '.1';
        }

        return $newVersion;
    }

    /**
     * Get the current version of a language package for a given project
     *
     * @param integer $projectId The id of the project
     * @param string  $language  The language to look for
     *
     * @return string
     *
     * @throws Exception
     *
     * @since 1.0.0
     */
    public static function getCurrentLanguagePackageVersion(int $projectId, string $language): string
    {
        $langPackVersion  = '';
        $session          = Factory::getApplication()->getSession();
        $langPackVersions = $session->get('prevLangPackVersions', null, 'BwTransifex');

        // First try to get language package version from session
        if (is_array($langPackVersions))
        {
            foreach ($langPackVersions as $item)
            {
                if ($item['lang_name'] === $language)
                {
                    $langPackVersion = $item['lang_version'];
                    break;
                }
            }
        }
        // If the session provides nothing, get language package version from table
        else
        {
            $db    = Factory::getContainer()->get('DatabaseDriver');
            $query = $db->getQuery(true);

            $query->select($db->quoteName('lang_version'));
            $query->from($db->quoteName('#__bwtransifex_zips'));
            $query->where($db->quoteName('project_id') . ' = ' . $db->Quote($projectId));
            $query->where($db->quoteName('lang_name') . ' = ' . $db->Quote($language));

            try
            {
                $db->setQuery($query);
                $langPackVersion = $db->loadResult();
            }
            catch (RuntimeException $e)
            {
                Factory::getApplication()->enqueueMessage($e->getMessage(), 'error');
            }
        }

        return (string)$langPackVersion;
    }

    /**
     * Method to explicitly set slashes at beginning and end of given path
     *
     * @param $path
     *
     * @return string
     *
     * @since 1.0.0
     *
     */
    private static function cleanupPath($path): string
    {
        if ($path[0] !== '/')
        {
            $path = '/' . $path;
        }

        $lastChar = strlen($path) - 1;

        if ($path[$lastChar] !== '/')
        {
            $path = $path . '/';
        }

        return $path;
    }

    /**
     * Method to store old BwTransifex status to session to be able to only pack changed languages
     *
     * @param int    $projectId   id of the project
     * @param string $sessionName name opf session part
     *
     * @return void
     *
     * @throws Exception
     *
     * @since 1.0.0
     *
     */
    public static function storeLangStatsAtSession(int $projectId, string $sessionName): void
    {
        $session = Factory::getApplication()->getSession();

        $languages = $session->get('usedLanguages', null, 'bwtransifex');
        $previousTxStatus = self::getTxStatusFromDb($projectId, $languages);

        $session->set($sessionName, $previousTxStatus, 'bwtransifex');
    }

    /**
     * Method to check if there are new translations for given language
     *
     * @param array $previousTxStatus previous update dates
     * @param array $currentTxStatus  current update dates
     *
     * @return boolean
     *
     * @since 1.0.0
     *
     */
    public static function hasChangedStrings(array $previousTxStatus, array $currentTxStatus): bool
    {
        if (count($previousTxStatus) != count($currentTxStatus))
        {
            return true;
        }

        $uniqueElements = array_merge(array_diff_key($previousTxStatus, $currentTxStatus), array_diff_key($currentTxStatus, $previousTxStatus));

        if (count($uniqueElements))
        {
            return true;
        }

        foreach ($previousTxStatus as $key => $value)
        {
            if ($currentTxStatus[$key] != $value)
            {
                return true;
            }
        }

        return false;
    }

    /**
     * Method to get BwTransifex status of translations (raw_data) from table languages for specified languages
     *
     * @param int   $projectId id of project
     * @param array $languages array of languages
     *
     * @return array over first languages and second resource name holding raw translation data
     *
     * @since 1.0.0
     *
     */
    public static function getTxStatusFromDb(int $projectId, array $languages): array
    {
        $langStatus = array();
        $db         = Factory::getContainer()->get('DatabaseDriver');

        foreach ($languages as $language)
        {
            $query = $db->getQuery(true);

            $query->select($db->quoteName('r') . '.' . $db->quoteName('resource_name'));
            $query->select($db->quoteName('l') . '.' . $db->quoteName('raw_data'));
            $query->from($db->quoteName('#__bwtransifex_languages') . ' as '. $db->quoteName('l'));
            $query->from($db->quoteName('#__bwtransifex_resources') . ' as '. $db->quoteName('r'));
            $query->where($db->quoteName('l') . '.' . $db->quoteName('project_id') . ' = ' . $db->quote($projectId));
            $query->where($db->quoteName('l') . '.' . $db->quoteName('lang_name') . ' = ' . $db->quote($language));
            $query->where($db->quoteName('r') . '.' . $db->quoteName('id') . ' = ' . $db->quoteName('l') . '.' . $db->quoteName('resource_id'));

            $db->setQuery($query);

            $currentStatus = $db->loadAssocList();

            foreach ($currentStatus as $status)
            {
                $statusObj = json_decode($status['raw_data']);
                $langStatus[$language][$status['resource_name']] = $statusObj->last_update;
            }
        }

        return $langStatus;
    }

    /**
     * Method to get used languages from database
     *
     * @param $projectId
     *
     * @return array
     *
     * @since 1.0.0
     *
     */
    public static function getUsedLanguagesFromDb($projectId): array
    {
        $db    = Factory::getContainer()->get('DatabaseDriver');
        $query = $db->getQuery(true);

        $query->select("DISTINCT " . $db->quoteName('lang_name'));
        $query->from($db->quoteName('#__bwtransifex_languages'));
        $query->where($db->quoteName('project_id') . '=' . $db->quote($projectId));
        $query->order($db->quoteName('lang_name'));

        $db->setQuery($query);

        return (array) $db->loadColumn();
    }

    /**
     * Method to replace variables in files
     *
     * @param string $fileData content of file
     * @param object $project  the project
     * @param string $language language of this package
     *
     * @return string
     *
     * @throws Exception
     *
     * @since 1.0.0
     *
     */
    public static function replaceVariables(string $fileData, object $project, string $language): string
    {
        $year            = date('Y');
        $date            = date('Y-m-d');
        $time            = date('H-i-s');
        $langPackVersion= self::getNewLanguagePackageVersion($project->id, $language);

        $replaceConfig = $project->params['replace_config'];
        $replaceLines = explode("\r\n", $replaceConfig);

        foreach($replaceLines as $row)
        {
            $line = explode('=', $row);

            if (str_contains($line[1], '@@'))
            {
                $item = str_replace('@@VERSION@@', $langPackVersion, $line[1]);
                $item = str_replace('@@YEAR@@', $year, $item);
                $item = str_replace('@@DATE@@', $date, $item);
                $item = str_replace('@@TIME@@', $time, $item);

                $replaceArray[$line[0]] = $item;
            }
            else
            {
                $replaceArray[$line[0]] = $line[1];
            }
        }

        $replaceArray['"_QQ_"'] = "'";

        $replaceValues = array(
            '1.0.1' => $langPackVersion,
            '2025' => $year,
//			'"_QQ_"' => "'",
        );

        foreach ($replaceArray as $key => $value)
        {
            $fileData = str_replace($key, $value, $fileData);
        }

        return $fileData;
    }
}
