<?php
/**
 * BwTransifex Component
 *
 * BwTransifex download controller class for the component backend
 *
 * @version 1.0.1
 * @package BwTransifex
 * @subpackage BwTransifex Component Site
 * @author Romana Boldt
 * @copyright (C) 2025 Boldt Webservice <forum@boldt-webservice.de>
 * @support https://www.boldt-webservice.de/en/forum-en/forum/bwtransifex.html
 * @license GNU/GPL, see LICENSE.txt
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

namespace BoldtWebservice\Component\BwTransifex\Site\Controller;

// No direct access
defined('_JEXEC') or die('Restricted access');

use Exception;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\MVC\Controller\BaseController;
use BoldtWebservice\Component\BwTransifex\Administrator\Helper\BwTransifexHelperPackage;
use Joomla\CMS\MVC\Model\BaseDatabaseModel;
use Joomla\CMS\Router\Route;
use stdClass;

/**
 * Define the BwTransifex download controller class
 *
 * @package BwTransifex Site
 *
 * @since 1.0.0
 */
class DownloadController extends BaseController
{
    /**
     * Constructor.
     *
     * @param	array	$config		An optional associative array of configuration settings.
     *
     * @throws Exception
     *
     * @since 1.0.0
     *
     * @see		JController
     */
    public function __construct($config = array())
    {
        parent::__construct($config, $this->factory);

        $this->factory = $this->app->bootComponent('com_bwtransifex')->getMVCFactory();

        // Register Extra tasks
        $this->registerTask('applyNew', 'saveNew');
        $this->registerTask('save2New', 'saveNew');
        $this->registerTask('add', 'add');
    }

    /**
     * Proxy for getModel.
     *
     * @param string $name   The name of the model.
     * @param string $prefix The prefix for the PHP class name.
     * @param array  $config An optional associative array of configuration settings.
     *
     * @return BaseDatabaseModel
     *
     * @since 1.0.0
     */
    public function getModel($name = 'Project', $prefix = 'Administrator', $config = array('ignore_request' => true)): BaseDatabaseModel
    {
        return $this->factory->createModel($name, $prefix, $config);
    }

    /**
     * The download function
     *
     * @throws Exception
     *
     * @return void
     *
     * @since 1,0,0
     */
    public function download(): void
    {
        $input     = Factory::getApplication()->input;
        $language  = $input->getInt('bwlanguage', 0);
        $projectId = $input->getInt('id', 0);
        $result    = new stdClass();

        $db = Factory::getContainer()->get('DatabaseDriver');

        $query = $db->getQuery(true);

        // Get the entry for provided project and provided language from table
        $query->select('z.*, p.transifex_slug, p.extension_name, z.id, z.lang_name')
            ->from($db->qn('#__bwtransifex_zips') . ' AS z')
            ->leftJoin($db->qn('#__bwtransifex_projects') . ' AS p ON ' . $db->qn('p.id') . '=' . $db->qn('z.project_id'))
            ->where($db->qn('z.id') . ' = ' . $db->q($language));

        try
        {
            $db->setQuery($query, 0, 1);
            $result = $db->loadObject();

            // If an entry is found, provide path for downloading
            if ($result)
            {
                // Calculate the path of the language package
                $languagePackVersion = BwTransifexHelperPackage::getCurrentLanguagePackageVersion($result->project_id, $result->lang_name);
                $goodSlug = str_replace(' ', '_', $result->extension_name);

                $filepath = JPATH_ROOT . '/media/com_bwtransifex/packages/' . $result->transifex_slug . '/'
                    . $result->lang_name . '/' . $result->lang_name . '.' . $goodSlug . '.' . $languagePackVersion . '.zip';

                // If the language package exists, provide the download path
                if (file_exists($filepath) === true) {
                    // Set example variables
                    $filename = $result->lang_name . '.' . $goodSlug . '.' . $languagePackVersion . '.zip';

                    // Http headers for zip downloads
                    header("Pragma: public");
                    header("Expires: 0");
                    header("Cache-Control: must-revalidate, post-check=0, pre-check=0");
                    header("Cache-Control: public");
                    header("Content-Description: File Transfer");
                    header("Content-type: application/zip");
                    header("Content-Disposition: attachment; filename=\"" . $filename . "\"");
                    header("Content-Transfer-Encoding: binary");
                    header("Content-Length: " . filesize($filepath));
                    ob_end_flush();
                    readfile($filepath);
                    Factory::getApplication()->close();
                }
                // If the language package does not exist, provide error message and redirect user to the language file download page
                else
                {
                    Factory::getApplication()->enqueueMessage(Text::sprintf('COM_BWTRANSIFEX_DOWNLOAD_LANGUAGE_PACKAGE_DOES_NOT_EXIST', $result->lang_name), 'error');
                    $this->setRedirect(
                        Route::_(
                            'index.php?option=com_bwtransifex&view=language&zip=' . $result->id . '&project=' . $result->project_id,
                            false
                        )
                    );
                }
            }
            else
            {
                // If an entry for this project with this language does not exist, provide error message and redirect
                // user to single project page
                Factory::getApplication()->enqueueMessage(Text::sprintf('COM_BWTRANSIFEX_DOWNLOAD_LANGUAGE_PACKAGE_TABLE_ENTRY_NOT_EXIST', $result->lang_name), 'error');
                $this->setRedirect(
                    Route::_(
                        'index.php?option=com_bwtransifex&view=project&id=' . $projectId,
                        false
                    )
                );
            }
        }
        catch (Exception $e)
        {
            Factory::getApplication()->enqueueMessage(Text::sprintf('COM_BWTRANSIFEX_DOWNLOAD_LANGUAGE_PACKAGE_TABLE_ERROR', $result->lang_name), 'error');
            $this->setRedirect(
                Route::_(
                    'index.php?option=com_bwtransifex&view=project&id=' . $projectId,
                    false
                )
            );
        }
    }
}
